import os
import random
import carla
import copy
import weakref
import numpy as np
import threading
import time
import queue
from copy import deepcopy

from srunner.scenariomanager.carla_data_provider import CarlaDataProvider
from comm.channel import Channel, LockBuffer
from comm.message_types import Beacon, Language, Representation, Topic

import comm.comm_config as comm_config

class VehicleState(object):
    def __init__(self):
        self.lock = threading.Lock()
        self.FrameID = -1
        self.myTrans = None
        self.speed = None
        self.PeerList_LangChannel = []
        self.PeerList_BeaconChannel = []
        self.PeerList_ReprChannel = []

    def get_state(self): # use deepcopy
        self.lock.acquire()
        state = VehicleState()
        state.FrameID = self.FrameID
        state.myTrans = self.myTrans
        state.speed = self.speed
        state.PeerList_LangChannel = self.PeerList_LangChannel
        state.PeerList_BeaconChannel = self.PeerList_BeaconChannel
        state.PeerList_ReprChannel = self.PeerList_ReprChannel
        self.lock.release()
        return state

    def set_state(self, _fid, _my_transform, _speed, _beacon_peers, _lang_peers, _repr_peers):
        self.lock.acquire()
        self.FrameID = _fid
        self.myTrans = _my_transform
        self.speed = _speed
        self.PeerList_LangChannel = _lang_peers
        self.PeerList_BeaconChannel = _beacon_peers
        self.PeerList_ReprChannel = _repr_peers
        self.lock.release()


class Transceiver(object):
    """
    Transceiver is a interface for Vehicle to communicate with other vehicles
    You can configure multiple channels for the Transceiver
    Transceiver must have a beacon channel to share meta data
    """
    def __init__(self,
                 vehicle,
                 agent,
                 sharing_mode=False,
                 channels=[Topic.BEACON, Topic.LANGUAGE, Topic.REPRESENTATION],
                 ):
        self.vehicle = vehicle
        self.id = self.vehicle.id
        self.agent_id = agent.agent_id
        self.agent = agent
        self.world = CarlaDataProvider.get_world()
        self.state = VehicleState()
        self.startFrameID = -1
        self.ProcessThread = None

        """Comm"""
        self.sharing_mode = sharing_mode
        self.beacon_channel = Channel(self, Topic.BEACON)
        self.lang_channel = Channel(self, Topic.LANGUAGE)
        self.repr_channel = Channel(self, Topic.REPRESENTATION)

    def destroy(self):
        self.beacon_channel.destroy()
        self.lang_channel.destroy()
        if self.ProcessThread is not None:
            self.ProcessThread.join()

    def update_vehicle_state(self, frameID):
        self.state.set_state(frameID,
                             self.vehicle.get_transform(),
                             self.vehicle.get_velocity(),
                             list(self.beacon_channel.buffer.keys()),
                             list(self.lang_channel.buffer.keys()),
                             list(self.repr_channel.buffer.keys()),
                             )
        return self.state.get_state()

    def TX_Language(self, message):
        """
        Publish a message to the channel
        """
        mState = self.update_vehicle_state(self.world.get_snapshot().timestamp.frame)
        myTrans = mState.myTrans
        m = Language(self.id, self.agent_id, mState.FrameID,
                     myTrans.location.x, myTrans.location.y, myTrans.location.z,
                     myTrans.rotation.yaw, myTrans.rotation.pitch, myTrans.rotation.roll,
                     message)
        self.lang_channel.publish(m)

    def get_lang_data(self):
        """
        Get language data from the channel
        """
        message_data = {}
        for peerId, buffer in self.lang_channel.buffer.items():
            if int(peerId) == int(self.id):
                continue
            message = buffer.get_content(comm_config.MaxDataAge)
            if message is not None:
                message_data[peerId] = message
        return message_data

    def TX_Representation(self, message):
        """
        Publish a representation message to the channel
        """
        mState = self.update_vehicle_state(self.world.get_snapshot().timestamp.frame)
        myTrans = mState.myTrans
        m = Representation(self.id, self.agent_id, mState.FrameID,
                           myTrans.location.x, myTrans.location.y, myTrans.location.z,
                           myTrans.rotation.yaw, myTrans.rotation.pitch, myTrans.rotation.roll,
                           message['representation'], message['xyz'], message['transform'])
        self.repr_channel.publish(m)

    def get_repr_data(self):
        """
        Get representation data from the channel
        """
        message_data = {}
        for peerId, buffer in self.repr_channel.buffer.items():
            if int(peerId) == int(self.id):
                continue
            message_data[peerId] = buffer.get_content()
        return message_data
